/**
* \file: PluginManager.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: CarPlay
*
* \author: J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
*
* \copyright (c) 2013-2014 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include "Common.h"
#include "PluginManager.h"
#include <dlfcn.h>
#include <unistd.h>

namespace adit { namespace carplay
{

using namespace std;

typedef void (*_LibraryEntryPointFn)();
typedef void (*carplay_InitConfiguration_t)(adit::carplay::IDynamicConfiguration* inConfig);

PluginManager::~PluginManager()
{
    for (auto library : libraries)
    {
        LOGD_DEBUG((dipo, "unload plugin %s", library.Name.c_str()));
        auto exitPoint = (_LibraryEntryPointFn)dlsym(library.Handle, "carplay_LibraryExitPoint");
        auto error = dlerror();
        if (error == nullptr && exitPoint != nullptr)
        {
            exitPoint();
        }
        else
        {
            // it is allowed to not have the exit point, ignore errors
            LOGD_DEBUG((dipo, "no carplay_LibraryExitPoint function in %s", library.Name.c_str()));
        }

        //dlclose(library.Handle);
    }
    libraries.clear();
}

bool PluginManager::Initialize(IDynamicConfiguration& inConfig)
{
    auto plugins = inConfig.GetItems("plugins");
    for (auto plugin : plugins)
    {
        string library = plugin;
        load(library, inConfig);
    }

    if (libraries.empty())
    {
        LOG_ERROR((dipo, "Could not find/load any plug-in adapter libraries!"));
        return false;
    }

    return true;
}

/* ====== private functions ====== */

void PluginManager::load(const string& inName, IDynamicConfiguration& inConfig)
{
    LOGD_DEBUG((dipo, "load plugin %s", inName.c_str()));

    Library lib;
    lib.Handle = dlopen(inName.c_str(), RTLD_NOW);
    char* error = dlerror();

    if (error != nullptr || lib.Handle == nullptr)
    {
        LOG_ERROR((dipo, "load plugin %s failed: %s", inName.c_str(), error));
        return;
    }

    auto entryPoint = (_LibraryEntryPointFn)dlsym(lib.Handle, "carplay_LibraryEntryPoint");
    error = dlerror();
    if (error != nullptr || entryPoint == nullptr)
    {
        dlclose(lib.Handle);
        LOG_ERROR((dipo, "load plugin %s failed: %s", inName.c_str(), error));
        return;
    }

    entryPoint();

    auto initConfiguration = (carplay_InitConfiguration_t)dlsym(lib.Handle,
            "carplay_InitConfiguration");
    error = dlerror();
    if (error == nullptr && initConfiguration != nullptr)
    {
        LOGD_DEBUG((dipo, "execute carplay_InitConfiguration of %s", inName.c_str()));
        initConfiguration(&inConfig);
    }
    else
    {
        LOGD_DEBUG((dipo, "%s contains no carplay_InitConfiguration", inName.c_str()));
    }

    lib.Name = inName;
    libraries.push_back(lib);
}

} } // namespace adit { namespace carplay
